 /*
   *  Object %name    : FVOS.c
   *  State           :  %state%
   *  Creation date   :  Wed Nov 17 17:39:24 2004
   *  Last modified   :  %modify_time%
   */
  /** @file
   *  \brief Secure Boot implemtation of hash queues
   *
   *  \version 
   *  \author yevgenys
   *  \remarks Copyright (C) 2004 by Discretix Technologies Ltd.
   *           All Rights reserved
   */

#include <linux/workqueue.h>
#include <linux/scatterlist.h>

#include "BaseTypes.h"
#include "FVOS_Types.h"
#include "FVOS_API.h"
#include "FVOS_Config.h"
#include "FVOS_Error.h"
#include "sep_driver_config.h"
#include "sep_driver_api.h"


/* for memcpy */
#include <linux/mm.h>

/*---------------------
    DEFINES
-------------------------*/

/* mask for checking the type of data */
#define FVOS_DATA_TYPE_MASK       0xC0000000UL

/* offset in bits of the data type */
#define FVOS_DATA_TYPE_OFFSET     30

/* type for scatter data type */
#define FVOS_DATA_TYPE_SCATTER    0x2 

/* type for user space virtual buffer */
#define FVOS_DATA_TYPE_USER       0x0

/* type for kernel virtual buffer */
#define FVOS_DATA_TYPE_KERNEL     0x1

/*-------------------------
  GLOBAL variables
--------------------------*/

extern DxUint32_t     sep_in_num_pages;

extern DxUint32_t     sep_out_num_pages;

extern struct page**  sep_in_page_array;

/* array of pointers to the pages that represent out data for the synchronic DMA action  */
extern struct page**  sep_out_page_array;

/* counter for the gpr value */
DxUint32_t          g_GPRPollValue;


/* address of the SEP shared area */
DxUint32_t          g_FVOS_SharedAreaAddress;

/* the start address of the area where the DMA tables for synchronic symmetric operations are created */
DxUint32_t          g_FVOS_DMATableAreaAddress;

/* allocation offset */
DxUint32_t          g_FVOS_AllocOffset = 0;

/* static memory address */
DxUint32_t          g_FVOS_StaticMemoryAddr;

/* start address of memory region for allocations */
DxUint32_t          g_FVOS_AllocMemoryAddr;

/* start adress of the region for the system data (time, caller id etc) */
DxUint32_t          g_FVOS_SystemDataMemoryAreaAddress;


/*-------------------------
  DEFINES
--------------------------*/


/*-----------------------------
    prototypes
-------------------------------*/

  


/**
 * @brief     This function handles the build DMa tables for symmetric action (AES,DES,HASH)
 *
 * @param[in/out] cmd - command structure wuth input/output argument
 * 
 * @return     DxError_t:  
 *                        
 */                                                
static FVOS_Error_t FVOS_HandleBuildSymDmaTablesCmd(FVOS_Cntrl_info_t* cmdData_ptr);
      



/**
 * @brief     This function constructs the lli table from the input of the array of physical addresses
 *            sizes for each physical buffer ,  and the number of physical buffer. Function uses the the VOS function for allocation of the
 *            memory needed for the 
 *
 *
 * @param[in] inputPhysicalBuffersArray - input physical buffers and their size
 * @param[out] firstInputLLITable_ptr - holds pointer to the first lli table in the list
 * @param[out] numEntriesInTable_ptr - number of entries in the first lli table 
 * 
 * @return     DxError_t:  
 *                        
 */
static FVOS_Error_t FVOS_PrepareInputDMATable(DxUint32_t   dataInAddr,
                                              DxUint32_t   dataInSize,
                                              DxUint32_t*  firstInputLLITable_ptr,
                                              DxUint32_t*  numEntriesInTable_ptr,
                                              DxUint32_t*  firstTableDataSize_ptr);


/**
 * @brief     This function creates tables for symmetric engines (AES,DES). The size of data in each
 *            table is modulud the operationBlockSize
 *
 *
 * @param[in] inputPhysicalBuffersArray - physical memory buffers for input
 * @param[in] outputPhysicalBuffersArray - physical memory buffers for output
 * @param[in] operationBlockSize - block size of the operation (AES or DES)
 * @param[out] firstInputLLITable_ptr - pointer to the first table in the list of input tables
 * @param[out] firstOutputLLITable_ptr - pointer to the first table in the list of output tables
 * @param[out] numInTableEntries_ptr - number of entries in the first input table
 * @param[out] numOutTableEntries_ptr - number of entries in the first output table
 * @param[out] firstTableDataSize_ptr - the size of data in the first table (input or output, since they must be equal)
 * 
 * @return     DxError_t:  
 *                        
 */
static FVOS_Error_t FVOS_PrepareInputOutputDMATables(DxUint32_t   dataInAddr,
                                                     DxUint32_t   dataOutAddr,
                                                     DxUint32_t   dataInSize,
                                                     DxUint32_t   operationBlockSize,
                                                     DxUint32_t*  firstInputLLITable_ptr,
                                                     DxUint32_t*  firstOutputLLITable_ptr,
                                                     DxUint32_t*  numInTableEntries_ptr,
                                                     DxUint32_t*  numOutTableEntries_ptr,
                                                     DxUint32_t*  firstTableDataSize_ptr);
                                                     
 /**
 * @brief     This function prepares in and out lli arrays from the given in and out scatter lists
 *
 *
 * @param[in] dataInAddr - pointer to the in scatter list
 * @param[in] dataOutAddr - pointer to the out scatter list
 * @param[out] lli_in_array_ptr -pointer to the allocated in lli array 
 * @param[out] lli_in_array_size_ptr - pointer to the number of entries in the lli array
 * @param[out] lli_out_array_ptr - pointer to the allocated out lli array 
 * @param[out] lli_out_array_size_ptr - pointer to the number of entries out the lli array
 
 * @return     DxError_t:  
 *                        
 */                                                    
static FVOS_Error_t FVOS_PrepareLLIScatterList(DxUint32_t         dataInAddr,
                                               DxUint32_t         dataOutAddr,
                                               sep_lli_entry_t**  lli_in_array_ptr,
                                               DxUint32_t*        lli_in_array_size_ptr,
                                               sep_lli_entry_t**  lli_out_array_ptr,
                                               DxUint32_t*        lli_out_array_size_ptr);
                                               
/**
 * @brief     This function prepares in and out lli arrays from the given in and out user space addresses
 *
 *
 * @param[in] dataInAddr - pointer to the input user space address
 * @param[in] dataOutAddr - pointer to the output user space address
 * @param[in] dataSize - size of data 
 * @param[out] lli_in_array_ptr -pointer to the allocated in lli array 
 * @param[out] lli_out_array_ptr - pointer to the allocated out lli array 
 
 * @return     DxError_t:  
 *                        
 */                                                    
static FVOS_Error_t FVOS_PrepareLLIUserSpace(DxUint32_t         dataInAddr,
                                             DxUint32_t         dataOutAddr,
                                             DxUint32_t         dataSize,
                                             sep_lli_entry_t**  lli_in_array_ptr,
                                             sep_lli_entry_t**  lli_out_array_ptr);
                                             
/**
 * @brief     This function prepares in and out lli arrays from the given in and out kernel space addresses
 *
 *
 * @param[in] dataInAddr - pointer to the input kernel space address
 * @param[in] dataOutAddr - pointer to the output kernel space address
 * @param[in] dataSize - size of data 
 * @param[out] lli_in_array_ptr -pointer to the allocated in lli array 
 * @param[out] lli_in_array_size_ptr - pointer to the number of entries in the lli array
 * @param[out] lli_out_array_ptr - pointer to the allocated out lli array 
 * @param[out] lli_out_array_size_ptr - pointer to the number of entries out the lli array
 
 * @return     DxError_t:  
 *                        
 */                                                    
static FVOS_Error_t FVOS_PrepareLLIKernelSpace(DxUint32_t         dataInAddr,
                                               DxUint32_t         dataOutAddr,
                                               DxUint32_t         dataSize,
                                               sep_lli_entry_t**  lli_in_array_ptr,
                                               DxUint32_t*        lli_in_array_size_ptr,
                                               sep_lli_entry_t**  lli_out_array_ptr,
                                               DxUint32_t*        lli_out_array_size_ptr);
                                               


/*--------------------------
    FUNCTIONS
----------------------------*/

/**
 * @brief     This functions configurates the FVOS - shared area address
 *
 * @param[in] sharedAreaAddr - address of the shared area
 * @return     DxError_t_t:  
 *                        
 */
FVOS_Error_t  FVOS_Config(DxUint32_t sharedAreaAddr)
{
 
  
  /*-------------------------
      CODE
  ----------------------------*/
  /* set the shared area */
  g_FVOS_SharedAreaAddress = sharedAreaAddr;
  
  /* set the static memory address at the end of the shared message area */
  g_FVOS_StaticMemoryAddr = sharedAreaAddr + FVOS_MESSAGE_MEMORY_SIZE_IN_BYTES;
  
  /* set the allocation memory address at the end of the static memory */
  g_FVOS_AllocMemoryAddr = g_FVOS_StaticMemoryAddr + FVOS_STATIC_MEMORY_SIZE_IN_BYTES;
  
  /* set the DMA tables start address at the end of the allocation memory area */
  g_FVOS_DMATableAreaAddress = g_FVOS_AllocMemoryAddr + FVOS_ALLOC_MEMORY_SIZE_IN_BYTES;
     
  

  return FVOS_OK;
}

/**
 * @brief     This functions start the transaction to the SEP - this is the first function to be called during transaction to SEP
 *            In NoOS configuration no action needs to be performed
 * 
 * @return     DxError_t_t:  
 *                        
 */
FVOS_Error_t FVOS_StartTransaction()
{
  /* error */
  FVOS_Error_t    error;
  /*-----------------------------
      CODE
  ---------------------------------*/
  
  error = sep_lock();
  
  return error;
}

/**
 * @brief     This function end the transaction to SEP - this is the last function in the transaction
 *            In NoOS configuration no action needs to be performed
 *
 * @return     DxError_t_t:  
 *                        
 */
FVOS_Error_t FVOS_EndTransaction(DxUint32_t mappedAddr)
{
    
    
  /*-----------------------------
      CODE
  ---------------------------------*/
  sep_unlock();
    
  return FVOS_OK;
}

/**
 * @brief     This function maps the shared message area to the application memory. It returns a pointer,
 *            trough which application can directly write message data into message shared area
 * 
 * @param[out] mappedAddr_ptr - the mapped address will be returned into this pointer
 *
 * @return     DxError_t_t:  
 *                        
 */
FVOS_Error_t FVOS_Mmap(DxUint32_t* mappedAddr_ptr)
{ 
  /*-----------------------------
      CODE
  ---------------------------------*/
  /* returns the address of the message shared area */
  sep_map_shared_area(mappedAddr_ptr);
    
  return FVOS_OK;
}


/**
 * @brief     This function performs a number of commands according to the CMD paramater:
 *            1) Signal to SEP that the message is ready
 *            2) Allocate memory continues locked memory
 *            3) Read from the allocated memory
 *            4) Write into allocated memory
 * 
 * @param[in] commandId - the id of command to execute
 * @param[in] cmdData_ptr - pointer to the strucutre that contains the paramaters for command execution
 *
 * @return     DxError_t_t:  
 *                        
 */
FVOS_Error_t FVOS_Cntrl(FVOS_Cntrl_cmd_t commandId , FVOS_Cntrl_info_t* cmdData_ptr)
{
  /* error */
 FVOS_Error_t    error;
  
  /*-----------------------------
      CODE
  ---------------------------------*/
  
  error = FVOS_OK;

  switch(commandId)
  {
    case FVOS_CNTRL_SEND_MSG_RDY_CMD:
      
      /* handle the send message to SEP command */      
      sep_send_msg_rdy_cmd();
      
      break;
    
    case FVOS_CNTRL_BUILD_SYM_DMA_TABLES_CMD:
    
     /* handle the command */
      error = FVOS_HandleBuildSymDmaTablesCmd(cmdData_ptr);
      
      break;
         
   
    case FVOS_CNTRL_FREE_DMA_RESOURSES:

      //error = sep_driver_HandleFreeDMAResources();
      /* no need for any action in the NoOS  */
      break;
      
    default:
      break;
  
  }

  
  return FVOS_OK;
}


/**
 * @brief     This function polls for the for the response from SEP
 *            Polling can be blocking or busy wait,and can be regulated by timeout
 * @param[in] Timeout - timeout for polling
 * @param[out] pollSource_ptr - the source of the poll interrupt (regular response, or MVS request from SEP
 * @return     DxError_t_t:  
 *                        
 */
FVOS_Error_t FVOS_Poll(DxUint32_t  timeout , DxUint32_t* pollSource_ptr)
{
  /* error */
  //FVOS_Error_t    error;
  
  //error = sep_driver_Poll(timeout,pollSource_ptr);
    
  return FVOS_OK;
}

/*-------------------------------------
    Private functions
---------------------------------------*/


/**
 * @brief     This function handles the allocate memory from dynamic pool command
 *
 * @param[in/out] cmd - command structure wuth input/output argument
 *
 * @return     DxError_t:  
 *                        
 */    
static FVOS_Error_t FVOS_HandleBuildSymDmaTablesCmd(FVOS_Cntrl_info_t* cmdData_ptr)
{
  /* error */
  FVOS_Error_t  error;

  /*--------------------------------
      CODE
  --------------------------------*/
  
  
  /* check if only input table must be built, or also output */
  if(cmdData_ptr->data.buildSymDMATables.dataOutAddr)
  {
    error = FVOS_PrepareInputOutputDMATables(cmdData_ptr->data.buildSymDMATables.dataInAddr,
                                             cmdData_ptr->data.buildSymDMATables.dataOutAddr,
                                             cmdData_ptr->data.buildSymDMATables.dataInSize,
                                             cmdData_ptr->data.buildSymDMATables.blockSize,
                                             &cmdData_ptr->data.buildSymDMATables.firstInTableAddr,
                                             &cmdData_ptr->data.buildSymDMATables.firstOutTableAddr,
                                             &cmdData_ptr->data.buildSymDMATables.firstInTableNumEntries,
                                             &cmdData_ptr->data.buildSymDMATables.firstOutTableNumEntries,
                                             &cmdData_ptr->data.buildSymDMATables.firstTableDataSize);
  }
  else
  {
    error = FVOS_PrepareInputDMATable(cmdData_ptr->data.buildSymDMATables.dataInAddr,
                                      cmdData_ptr->data.buildSymDMATables.dataInSize,
                                      &cmdData_ptr->data.buildSymDMATables.firstInTableAddr,
                                      &cmdData_ptr->data.buildSymDMATables.firstInTableNumEntries,
                                      &cmdData_ptr->data.buildSymDMATables.firstTableDataSize); 
  }
  
  return error;
}








/**
 * @brief     This function returns address of the stored time value 
 *
 * @param[out] timeAddress_ptr - address of the stored time value
 * 
 * @return     DxError_t:  
 *                        
 */
//static FVOS_Error_t FVOS_HandleCheckConfigurationCmd(FVOS_Cntrl_info_t* cmdData_ptr)
//{
  /*------------------
      CODE
  ----------------------*/
  
  //sep_driver_HandleCheckConfigurationCmd(cmdData_ptr);
  
  //return FVOS_OK;
//}

/**
 * @brief     This function constructs the lli table from the input of the array of physical addresses
 *            sizes for each physical buffer ,  and the number of physical buffer. Function uses the the VOS function for allocation of the
 *            memory needed for the 
 *
 *
 * @param[in] inputPhysicalBuffersArray - input physical buffers and their size
 * @param[out] firstInputLLITable_ptr - holds pointer to the first lli table in the list
 * @param[out] numEntriesInTable_ptr - number of entries in the first lli table 
 * 
 * @return     DxError_t:  
 *                        
 */
static FVOS_Error_t FVOS_PrepareInputDMATable(DxUint32_t   dataInAddr,
                                              DxUint32_t   dataInSize,
                                              DxUint32_t*  firstInputLLITable_ptr,
                                              DxUint32_t*  numEntriesInTable_ptr,
                                              DxUint32_t*  firstTableDataSize_ptr)
{  
  /* lli entry ptr */
  //FVOS_LliEntry_t*    lliEntryPtr;
  
  /* error */
  //FVOS_Error_t        error;
  
  
  /*------------------------
      CODE
  --------------------------*/
  
  /* init variables */
  //error = FVOS_OK;
   /*
  sep_driver_HandleCheckConfigurationCmd(dataInAddr,
                                         dataInSize,
                                         firstInputLLITable_ptr,
                                         numEntriesInTable_ptr,
                                         firstTableDataSize_ptr);
  
    */
  return FVOS_OK;

}


/**
 * @brief     This function creates tables for symmetric engines (AES,DES). The size of data in each
 *            table is modulud the operationBlockSize
 *
 *
 * @param[in] inputPhysicalBuffersArray - physical memory buffers for input
 * @param[in] outputPhysicalBuffersArray - physical memory buffers for output
 * @param[in] operationBlockSize - block size of the operation (AES or DES)
 * @param[out] firstInputLLITable_ptr - pointer to the first table in the list of input tables
 * @param[out] firstOutputLLITable_ptr - pointer to the first table in the list of output tables
 * @param[out] numEntriesInTable_ptr - number of entries in the first input/output tables
 * 
 * @return     DxError_t:  
 *                        
 */
static FVOS_Error_t FVOS_PrepareInputOutputDMATables(DxUint32_t   dataInAddr,
                                                     DxUint32_t   dataOutAddr,
                                                     DxUint32_t   dataInSize,
                                                     DxUint32_t   operationBlockSize,
                                                     DxUint32_t*  firstInputLLITable_ptr,
                                                     DxUint32_t*  firstOutputLLITable_ptr,
                                                     DxUint32_t*  numInTableEntries_ptr,
                                                     DxUint32_t*  numOutTableEntries_ptr,
                                                     DxUint32_t*  firstTableDataSize_ptr)
{
  
  /* error */
  FVOS_Error_t          error = FVOS_OK;
   
  /* num of entries in the in array */
  DxUint32_t            lli_in_array_size;
  
  /* num of entries in the out array */
  DxUint32_t            lli_out_array_size;
     
  /* pointer to the lli type */    
  sep_lli_entry_t*      lli_in_array;
  sep_lli_entry_t*      lli_out_array;
  
  /*-------------------------------
    CODE
  --------------------------------*/
  
  lli_out_array_size = lli_in_array_size = 0;
  
  switch( (dataInSize & FVOS_DATA_TYPE_MASK) >> FVOS_DATA_TYPE_OFFSET )
  {
    case FVOS_DATA_TYPE_USER:
    
      error = FVOS_PrepareLLIUserSpace(dataInAddr,
                                       dataOutAddr,
                                       dataInSize,
                                       &lli_in_array,
                                       &lli_out_array);
      if(error)
      {
        goto end_function;
      }
      
      lli_in_array_size = sep_in_num_pages;
      lli_out_array_size = sep_out_num_pages;
      
      break;
                          
    case FVOS_DATA_TYPE_SCATTER: 
      
      error = FVOS_PrepareLLIScatterList(dataInAddr,
                                         dataOutAddr,
                                         &lli_in_array,
                                         &lli_in_array_size,
                                         &lli_out_array,
                                         &lli_out_array_size);
      if(error)
      {
        goto end_function;
      }
      
      break;   
    case FVOS_DATA_TYPE_KERNEL:
    
      error = FVOS_PrepareLLIKernelSpace(dataInAddr,
                                         dataOutAddr,
                                         dataInSize,
                                         &lli_in_array,
                                         &lli_in_array_size,
                                         &lli_out_array,
                                         &lli_out_array_size);
      if(error)
      {
        goto end_function;
      }
      
      break; 
    default:
    
      break;
  }
  
     
  error = sep_construct_dma_tables_from_lli(lli_in_array,
                                            lli_in_array_size,
                                            lli_out_array,
                                            lli_out_array_size,
                                            operationBlockSize,
                                            firstInputLLITable_ptr,
                                            firstOutputLLITable_ptr,
                                            numInTableEntries_ptr,
                                            numOutTableEntries_ptr,
                                            firstTableDataSize_ptr);
  
end_function:
  
  return error;
}

/**
 * @brief     This function prepares in and out lli arrays from the given in and out user space addresses
 *
 *
 * @param[in] dataInAddr - pointer to the input user space address
 * @param[in] dataOutAddr - pointer to the output user space address
 * @param[in] dataSize - size of data 
 * @param[out] lli_in_array_ptr -pointer to the allocated in lli array 
 * @param[out] lli_out_array_ptr - pointer to the allocated out lli array 
 *
 *
 * @return     DxError_t:  
 *                        
 */                                                    
static FVOS_Error_t FVOS_PrepareLLIUserSpace(DxUint32_t         dataInAddr,
                                             DxUint32_t         dataOutAddr,
                                             DxUint32_t         dataSize,
                                             sep_lli_entry_t**  lli_in_array_ptr,
                                             sep_lli_entry_t**  lli_out_array_ptr)
{
  /* error */
  FVOS_Error_t  error;
  
  /*-------------------------
      CODE
  ---------------------------*/
  
  /* lock input data */
  error = sep_lock_user_pages(dataInAddr ,
                              dataSize ,
                              &sep_in_num_pages ,
                              lli_in_array_ptr,
                              &sep_in_page_array);
  if(error)
  {
    goto end_function;     
  }
      
  /* lock output data */
  error = sep_lock_user_pages(dataOutAddr ,
                              dataSize ,
                              &sep_out_num_pages ,
                              lli_out_array_ptr,
                              &sep_out_page_array);
  if(!error)
  {
    goto end_function;  
  }
  
  /* free the locked pages of input if error */
  sep_free_dma_pages(sep_in_page_array , sep_in_num_pages , 0);
  
  /* free the lli array */
  kfree(*lli_in_array_ptr);

end_function:

  return error;  
}


/**
 * @brief     This function prepares in and out lli arrays from the given in and out scatter lists
 *
 *
 * @param[in] dataInAddr - pointer to the in scatter list
 * @param[in] dataOutAddr - pointer to the out scatter list
 * @param[out] lli_in_array_ptr -pointer to the allocated in lli array 
 * @param[out] lli_in_array_size_ptr - pointer to the number of entries in the lli array
 * @param[out] lli_out_array_ptr - pointer to the allocated out lli array 
 * @param[out] lli_out_array_size_ptr - pointer to the number of entries out the lli array
 
 * @return     DxError_t:  
 *                        
 */                                                    
static FVOS_Error_t FVOS_PrepareLLIScatterList(DxUint32_t         dataInAddr,
                                               DxUint32_t         dataOutAddr,
                                               sep_lli_entry_t**  lli_in_array_ptr,
                                               DxUint32_t*        lli_in_array_size_ptr,
                                               sep_lli_entry_t**  lli_out_array_ptr,
                                               DxUint32_t*        lli_out_array_size_ptr)
{
  /* count */ 
  DxUint32_t           count;
    
  /* pointer to scatterlist type */
  struct scatterlist*  curr_sg;
  
  /* pointer to the lli type */    
  sep_lli_entry_t*     lli_in_array;
  
  sep_lli_entry_t*     lli_out_array;
  
  /*------------------------------
      CODE
  ----------------------------------*/
  
  /* Loop over each sg element, following the pointer to a new list if necessary
  aim: count count the number of elements for the kmalloc */
  for (count = 0 , curr_sg = (struct scatterlist*)dataInAddr; curr_sg != DX_NULL; count++ , curr_sg = sg_next(curr_sg));
     
  /* allocate buffer for the lli_in_array */
  lli_in_array = (sep_lli_entry_t*) kmalloc(count * sizeof(sep_lli_entry_t) , GFP_KERNEL | GFP_DMA );
  *lli_in_array_size_ptr = count;
         
  /* Return physical address and size for each sg entry */    
  for (curr_sg = (struct scatterlist*)dataInAddr; curr_sg != DX_NULL; lli_in_array++, curr_sg= sg_next(curr_sg))
  {
    lli_in_array->physical_address = sg_phys(curr_sg);
    lli_in_array->block_size=curr_sg->length;       
  }
     
   
  for (count = 0 , curr_sg = (struct scatterlist*)dataOutAddr; curr_sg != DX_NULL; count++ , curr_sg = sg_next(curr_sg));
   
  /* allocate buffer for the lli_out_array */
  lli_out_array = (sep_lli_entry_t*)kmalloc (count * sizeof(sep_lli_entry_t),GFP_KERNEL | GFP_DMA );
  *lli_out_array_size_ptr = count;
    
  /* Return physical address and size of an sg entry */    
  for (curr_sg = (struct scatterlist*)dataOutAddr; curr_sg != DX_NULL; lli_out_array++ , curr_sg = sg_next(curr_sg))
  {
    lli_out_array->physical_address = sg_phys(curr_sg);
    lli_out_array->block_size=curr_sg->length;
  }
  
  *lli_in_array_ptr = lli_in_array;
  *lli_out_array_ptr = lli_out_array;
  
  return 0;
  
}

/**
 * @brief     This function prepares in and out lli arrays from the given in and out kernel space addresses
 *
 *
 * @param[in] dataInAddr - pointer to the input kernel space address
 * @param[in] dataOutAddr - pointer to the output kernel space address
 * @param[in] dataSize - size of data 
 * @param[out] lli_in_array_ptr -pointer to the allocated in lli array 
 * @param[out] lli_in_array_size_ptr - pointer to the number of entries in the lli array
 * @param[out] lli_out_array_ptr - pointer to the allocated out lli array 
 * @param[out] lli_out_array_size_ptr - pointer to the number of entries out the lli array
 
 * @return     DxError_t:  
 *                        
 */                                                    
static FVOS_Error_t FVOS_PrepareLLIKernelSpace(DxUint32_t         dataInAddr,
                                               DxUint32_t         dataOutAddr,
                                               DxUint32_t         dataSize,
                                               sep_lli_entry_t**  lli_in_array_ptr,
                                               DxUint32_t*        lli_in_array_size_ptr,
                                               sep_lli_entry_t**  lli_out_array_ptr,
                                               DxUint32_t*        lli_out_array_size_ptr)
{
  return 0;
}
